# Dir 4.15 Evaluation of floating-point expressions shall not lead to the undetected generation of infinities and NaNs

## Category
Required

## Applies to
C90, C99, C11

## Amplification
If the evaluation of an arithmetic floating-point expression or a call to a mathematical function can possibly generate an infinity or a NaN (not-a-number), then that result shall be adequately tested. For efficiency reasons, it is possible to exploit the implementation-defined propagation rules for infinities and NaNs to delay the test, provided no infinity or NaN reaches sections of code that have not been designed to handle them.

## Rationale
Operations on floating-point numbers can overflow, that is, generate results of magnitude too large to be represented in the considered format. For such cases, the widely-used IEC 60559 Standard has special representations for positive and negative infinity. While the possibility to denote infinities is useful in some contexts because it allows operations to continue after overflow has occurred, much care has to be exercised, because:
*   It is likely that an error has occurred that resulted in an approximation error of unknown magnitude;
*   Infinities can easily generate unwanted underflows;
*   Careless computation with infinities can cause the generation of NaNs, whose correct handling requires even more care.

Note: an infinity does not necessarily correspond to a large value: for instance, on an implementation that conforms to IEC 60559, a *float* computation whose exact result is slightly larger than `3.4e+38` will be rounded to infinity and `logf( infinity )` returns infinity even though `logf( 3.4e+38 )` is less than 89.

Some operations on floating-point numbers are invalid or undetermined, such as taking the square root of a negative number or dividing a zero by a zero. The IEC 60559 Standard has special representations for such exceptional, non-numerical results: they are called NaNs. While of course there are legitimate uses of NaNs, programming with NaNs requires exceptional care, for example:
*   In some implementations there are both “signalling” and “quiet” NaNs;
*   NaNs can be signed;
*   `x == x` is false if `x` is a NaN;
*   Some library functions (such as `hypot()`) can return a non-NaN even if one of the arguments is NaN.

For these reasons, projects that do use infinities or NaNs need to take measures ensuring such special representations are detected and handled before they propagate to areas of the systems that are not prepared to receive them.

## Example
```c
#include <math.h>

extern double   get_result();       /* Return value may be infinite
                                       ... or valid data            */
extern void     use_result( double c ); /* Not protected against NANs or 
                                           ... or infinities        */

void f( void ) 
{
  double a = get_result(); 
  
  use_result( a );  /* Non-compliant - a not tested */
  
  if ( !isnan( a ) )
  {
    use_result( a ); /* Non-compliant - a may be infinite */
  }

  if ( isfinite( a ) ) 
  {
    use_result( a ); /* Compliant */
  }
}
```

The following example shows that it is not necessary to check for NaNs or infinities at every step:

```c
void g( void ) 
{
  double a = get_result(); 

  double b = exp( -2.0 * a ); /* Compliant - exp() propagates infinities
                                 ... and NaNs as expected */

  double c = (a * (1.0 + b) - (1.0 - b)) / (a * a);
  /* Division can result in NaN, even if operands are not infinity or NaN */

  use_result( c ); /* Non-compliant - not protected against NaNs
                      ... or infinities */

  if ( isfinite( c ) ) 
  {
    use_result( c ); /* Compliant - protected against NaNs */
                     /* ... and infinities */
  }
}
```

## See also
Dir 1.1, Dir 4.1, Dir 4.7

---

Copyright The MISRA Consortium Limited © [Date - March 2025].