<h2>Why is this an issue?</h2>
<p>When you implement <code>IComparable</code> or <code>IComparable&lt;T&gt;</code> on a class you should also override <code>Equals(object)</code>
and overload the comparison operators (<code>==</code>, <code>!=</code>, <code>&lt;</code>, <code>&lt;=</code>, <code>&gt;</code>,
<code>&gt;=</code>). That’s because the CLR cannot automatically call your <code>CompareTo</code> implementation from <code>Equals(object)</code> or
from the base comparison operator implementations. Additionally, it is best practice to override <code>GetHashCode</code> along with
<code>Equals</code>.</p>
<p>This rule raises an issue when a class implements <code>IComparable</code> without also overriding <code>Equals(object)</code> and the comparison
operators.</p>
<h3>Noncompliant code example</h3>
<pre>
public class Foo: IComparable  // Noncompliant
{
  public int CompareTo(object obj) { /* ... */ }
}
</pre>
<h3>Compliant solution</h3>
<pre>
public class Foo: IComparable
{
  public int CompareTo(object obj) { /* ... */ }
  public override bool Equals(object obj)
  {
    var other = obj as Foo;
    if (object.ReferenceEquals(other, null))
    {
      return false;
    }
    return this.CompareTo(other) == 0;
  }
  public int GetHashCode() { /* ... */ }
  public static bool operator == (Foo left, Foo right)
  {
    if (object.ReferenceEquals(left, null))
    {
      return object.ReferenceEquals(right, null);
    }
    return left.Equals(right);
  }
  public static bool operator &gt; (Foo left, Foo right)
  {
    return Compare(left, right) &gt; 0;
  }
  public static bool operator &lt; (Foo left, Foo right)
  {
    return Compare(left, right) &lt; 0;
  }
  public static bool operator != (Foo left, Foo right)
  {
    return !(left == right);
  }
}
</pre>

