<h2>Why is this an issue?</h2>
<p><a href="https://learn.microsoft.com/en-us/dotnet/api/system.componentmodel.defaultvalueattribute">DefaultValue</a> does not make the compiler set
the default value, as its name may suggest. What you probably wanted to use is <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.defaultparametervalueattribute">DefaultParameterValue</a>.</p>
<p>The <a href="https://learn.microsoft.com/en-us/dotnet/api/system.componentmodel.defaultvalueattribute">DefaultValue</a> attribute from the
<code>System.ComponentModel</code> namespace, is sometimes used to declare a member’s default value. This can be used, for instance, by the reset
feature of a visual designer or by a code generator.</p>
<pre>
public void DoStuff([DefaultValue(4)] int i)
{
    // i is not automatically assigned 4
}
</pre>
<p>The <a href="https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.optionalattribute">Optional</a> attribute from the
<code>System.Runtime.InteropServices</code> namespace is sometimes used to indicate that a parameter is optional, as an alternative to the
language-specific construct.</p>
<pre>
public void DoStuff([Optional] int i)
{
    // i would be assigned default(int) = 0
}
</pre>
<p>The use of <code>[DefaultValue]</code> with <code>[Optional]</code> has no more effect than <code>[Optional]</code> alone. That’s because
<code>[DefaultValue]</code> doesn’t actually do anything; it merely indicates the intent for the value.</p>
<pre data-diff-id="1" data-diff-type="noncompliant">
class MyClass
{
    public void DoStuff([Optional][DefaultValue(4)] int i, int j = 5)  // Noncompliant
    {
        Console.WriteLine(i);
    }

    public static void Main()
    {
        new MyClass().DoStuff(); // prints 0, since [DefaultValue] doesn't actually set the default, and default(int) is used instead
    }
}
</pre>
<p>More than likely, <code>[DefaultValue]</code> was used in confusion instead of <code>[DefaultParameterValue]</code>, the language-agnostic version
of the default parameter initialization mechanism provided by C#.</p>
<pre data-diff-id="1" data-diff-type="compliant">
class MyClass
{
    public void DoStuff([Optional][DefaultParameterValue(4)] int i, int j = 5)
    {
        Console.WriteLine(i);
    }

    public static void Main()
    {
        new MyClass().DoStuff(); // prints 4
    }
}
</pre>
<p>Notice that you can’t use both <code>[DefaultParameterValue]</code> and default parameter initialization on the same parameter.</p>
<pre>
void DoStuff([Optional][DefaultParameterValue(4)] int i = 5) // Error CS1745 Cannot specify default parameter value in conjunction with DefaultParameterAttribute or OptionalAttribute
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.optionalattribute">OptionalAttribute Class</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.componentmodel.defaultvalueattribute">DefaultValueAttribute Class</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.defaultparametervalueattribute">DefaultParameterValueAttribute
  Class</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/named-and-optional-arguments#optional-arguments">Optional arguments (C# Programming Guide)</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://stackoverflow.com/questions/40171095/use-optional-defaultparametervalue-attribute-or-not">Stack Overflow - Use "Optional,
  DefaultParameterValue" attribute, or not?</a> </li>
</ul>

