<h2>Why is this an issue?</h2>
<p>Non-static initializers, also known as instance initializers, are blocks of code within a class that are executed when an instance of the class is
created. They are executed when an object of the class is created just before the constructor is called. Non-static initializers are useful when you
want to perform some common initialization logic for all objects of a class. They allow you to initialize instance variables in a concise and
centralized manner, without having to repeat the same initialization code in each constructor.</p>
<p>While non-static initializers may have some limited use cases, they are rarely used and can be confusing for most developers because they only run
when new class instances are created.</p>
<h2>How to fix it</h2>
<p>Non-static initializers should be refactored into standard constructors or field initializers when possible.</p>
<p>In most cases, the use of constructors, overloaded constructors, or factory methods is preferable for initializing instance variables. These
approaches provide more explicit and controlled initialization, separate concerns, allow for better error handling, and make the code easier to
understand and maintain.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
class MyClass {
  private static final Map<String, String> MY_MAP = new HashMap<String, String>() {
    {
      put("a", "b");
    }
  }; // Noncompliant - HashMap should be extended only to add behavior, not for initialization
}
```
<h4>Compliant solution</h4>
<p>Using static initialization block:</p>

```java
class MyClass {
  private static final Map<String, String> MY_MAP = new HashMap<>();

  static {
    MY_MAP.put("a", "b");  // Compliant
  }
}
```
<p>or using constructor:</p>

```java
class MyClass {
  private static final Map<String, String> MY_MAP = new HashMap<>();

  public MyClass() {
    MY_MAP.put("a", "b");  // Compliant
  }
}
```
<p>or using Java 9 <code>Map.of</code>:</p>

```java
class MyClass {
  private static final Map<String, String> MY_MAP = java.util.Map.of("a", "b");   // Compliant
}
```
<p>or using Guava <code>ImmutableMap.of</code>:</p>

```java
class MyClass {
  private static final Map<String, String> MY_MAP = com.google.common.collect.ImmutableMap.of("a", "b");   // Compliant
}
```
<h2>Resources</h2>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.baeldung.com/java-static-instance-initializer-blocks">Static vs. Instance Initializer Block in Java</a> </li>
</ul>