<h2>Why is this an issue?</h2>
<p>Double-checked locking is the practice of checking a lazy-initialized object’s state both before and after a <code>synchronized</code> block is
entered to determine whether to initialize the object. In early JVM versions, synchronizing entire methods was not performant, which sometimes caused
this practice to be used in its place.</p>
<p>Apart from <code>float</code> and <code>int</code> types, this practice does not work reliably in a platform-independent manner without additional
synchronization of mutable instances. Using double-checked locking for the lazy initialization of any other type of primitive or mutable object risks
a second thread using an uninitialized or partially initialized member while the first thread is still creating it. The results can be unexpected,
potentially even causing the application to crash.</p>
<h2>How to fix it</h2>
<p>Given significant performance improvements of <code>synchronized</code> methods in recent JVM versions, <code>synchronized</code> methods are now
preferred over the less robust double-checked locking.</p>
<p>If marking the entire method as <code>synchronized</code> is not an option, consider using an inner <code>static class</code> to hold the reference
instead. Inner static classes are guaranteed to be initialized lazily.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public class ResourceFactory {
    private static Resource resource;

    public static Resource getInstance() {
        if (resource == null) {
            synchronized (DoubleCheckedLocking.class) { // Noncompliant, not thread-safe due to the use of double-checked locking.
                if (resource == null)
                    resource = new Resource();
            }
        }
        return resource;
    }
}
```
<h4>Compliant solution</h4>

```java
public class ResourceFactory {
    private static Resource resource;

    public static synchronized Resource getInstance() { // Compliant, the entire method is synchronized and hence thread-safe
        if (resource == null)
            resource = new Resource();
        return resource;
    }
}
```
<h4>Compliant solution</h4>
<p>Alternatively, a static inner class can be used. However, this solution is less explicit in its intention and hence should be used with care.</p>

```java
public class ResourceFactory {
    private static class ResourceHolder {
        public static Resource resource = new Resource(); // Compliant, as this will be lazily initialised by the JVM
    }

    public static Resource getResource() {
        return ResourceFactory.ResourceHolder.resource;
    }
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://www.cs.umd.edu/~pugh/java/memoryModel/DoubleCheckedLocking.html">The "Double-Checked Locking is Broken" Declaration</a> </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/6zdGBQ">CERT, LCK10-J.</a> - Use a correct form of the double-checked locking idiom </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/609">CWE-609 - Double-checked locking</a> </li>
  <li> <a href="https://docs.oracle.com/javase/specs/jls/se7/html/jls-12.html#jls-12.4">JLS 12.4</a> - Initialization of Classes and Interfaces </li>
  <li> Wikipedia: <a href="https://en.wikipedia.org/wiki/Double-checked_locking#Usage_in_Java">Double-checked locking</a> </li>
</ul>