<p>This rule raises an issue when a class overrides the <code>Object.clone</code> method instead of resorting to a copy constructor or other copy
mechanisms.</p>
<h2>Why is this an issue?</h2>
<p>The <code>Object.clone</code> / <code>java.lang.Cloneable</code> mechanism in Java should be considered broken for the following reasons and
should, consequently, not be used:</p>
<ul>
  <li> <code>Cloneable</code> is a <em>marker interface</em> without API but with a contract about class behavior that the compiler cannot enforce.
  This is a bad practice. </li>
  <li> Classes are instantiated without calling their constructor, so possible preconditions cannot be enforced. </li>
  <li> There are implementation flaws by design when overriding <code>Object.clone</code>, like type casts or the handling of
  <code>CloneNotSupportedException</code> exceptions. </li>
</ul>
<h2>How to fix it</h2>
<p>A copy constructor, copy factory or a custom copy function are suitable alternatives to the <code>Object.clone</code> /
<code>java.lang.Cloneable</code> mechanism.</p>
<p>Consider the following example:</p>

```java
class Entity implements Cloneable { // Noncompliant, using `Cloneable`

  public int value;
  public List<Entity> children; // deep copy wanted

  Entity() {
    EntityManager.register(this); // invariant
  }

  @Override
  public Entity clone() {
    try {
      Entity copy = (Entity) super.clone(); // invariant not enforced, because no constructor is caled
      copy.children = children.stream().map(Entity::clone).toList();
      return copy;
    } catch (CloneNotSupportedException e) { // this will not happen due to behavioral contract
      throw new AssertionError();
    }
  }
}
```
<p>The <code>Cloneable</code> / <code>Object.clone</code> mechanism could easily be replaced by copy constructor:</p>

```java
class Entity { // Compliant

  public int value;
  public List<Entity> children; // deep copy wanted

  Entity() {
    EntityManager.register(this); // invariant
  }

  Entity(Entity template) {
    value = template.value;
    children = template.children.stream().map(Entity::new).toList();
  }
}
```
<p>Or by a factory method:</p>

```java
class Entity { // Compliant

  public int value;
  public List<Entity> children; // deep copy wanted

  Entity() {
    EntityManager.register(this); // invariant
  }

  public static Entity create(Entity template) {
    Entity entity = new Entity();
    entity.value = template.value;
    entity.children = template.children.stream().map(Entity::new).toList();
    return Entity;
  }
}
```
<p>Or by a custom <code>copy</code> function:</p>

```java
class Entity { // Compliant

  public int value;
  public List<Entity> children; // deep copy wanted

  Entity() {
    EntityManager.register(this); // invariant
  }

  public Entity copy() {
    Entity entity = new Entity();
    entity.value = value;
    entity.children = children.stream().map(Entity::new).toList();
    return Entity;
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://www.artima.com/intv/bloch13.html">Joshua Bloch - Copy Constructor versus Cloning</a> </li>
  <li> <a href="https://docs.oracle.com/javase/8/docs/api/java/lang/Cloneable.html">Interface Cloneable - Java™ Platform, Standard Edition 8 API
  Specification</a> </li>
  <li> <a href="https://docs.oracle.com/javase/8/docs/api/java/lang/Object.html#clone--">Object.clone - Java™ Platform, Standard Edition 8 API
  Specification</a> </li>
</ul>
<h3>Related rules</h3>
<ul>
  <li> <a href='checks/checks/?languages=JAVA&tools=SONAR_LINT&rule=java:S2157'>java:S2157</a> - "Cloneables" should implement "clone" </li>
  <li> <a href='checks/checks/?languages=JAVA&tools=SONAR_LINT&rule=java:S1182'>java:S1182</a> - Classes that override "clone" should be "Cloneable" and call "super.clone()" </li>
</ul>