<h2>Why is this an issue?</h2>
<p>The <code>alt</code>, <code>aria-label</code> and <code>aria-labelledby</code> attributes provide a textual alternative to an image.</p>
<p>It is used whenever the actual image cannot be rendered.</p>
<p>Common reasons for that include:</p>
<ul>
  <li> The image can no longer be found </li>
  <li> Visually impaired users using a screen reader software </li>
  <li> Image loading is disabled, to reduce data consumption on mobile phones </li>
</ul>
<p>It is also very important not to set an alternative text attribute to a non-informative value. For example, <code>&lt;img ... alt="logo"&gt;</code>
is useless as it doesn’t give any information to the user. In this case, as for any other decorative image, it is better to use a CSS background image
instead of an <code>&lt;img&gt;</code> tag. If using CSS <code>background-image</code> is not possible, an empty <code>alt=""</code> is tolerated. See
Exceptions below.</p>
<p>This rule raises an issue when:</p>
<ul>
  <li> An <code>&lt;img&gt;</code> element has no <code>alt</code> attribute. </li>
  <li> An <code>&lt;input type="image"&gt;</code> element has no <code>alt</code>, <code>aria-label</code> or <code>aria-labelledby</code> attribute
  or they hold an empty string. </li>
  <li> An <code>&lt;area&gt;</code> element within an image map has no <code>alt</code>, <code>aria-label</code> or <code>aria-labelledby</code>
  attribute. </li>
  <li> An <code>&lt;object&gt;</code> element has no inner text, <code>title</code>, <code>aria-label</code> or <code>aria-labelledby</code>
  attribute. </li>
</ul>
<h3>Exceptions</h3>
<p><code>&lt;img&gt;</code> elements with an empty string&nbsp;<code>alt=""</code> attribute won’t raise any issue. However, this way should be used
in two cases only:</p>
<p>When the image is decorative and it is not possible to use a CSS background image. For example, when the decorative <code>&lt;img&gt;</code> is
generated via javascript with a source image coming from a database, it is better to use an <code>&lt;img alt=""&gt;</code> tag rather than generate
CSS code.</p>

```js
<li *ngFor="let image of images">
    <img [src]="image" alt="">
</li>
```
<p>When the image is not decorative but its <code>alt</code> text would repeat a nearby text. For example, images contained in links should not
duplicate the link’s text in their <code>alt</code> attribute, as it would make the screen reader repeat the text twice.</p>

```js
<a href="flowers.html">
    <img src="tulip.gif" alt="" />
    A blooming tulip
</a>
```
<p>In all other cases you should use CSS background images.</p>
<h2>How to fix it</h2>
<p>Add an alternative text to the HTML element.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
<img src="foo.png" /> <!-- missing `alt` attribute -->
<input type="image" src="bar.png" /> <!-- missing alternative text attribute -->
<input type="image" src="bar.png" alt="" /> <!-- empty alternative text attribute on <input> -->

<img src="house.gif" usemap="#map1"
    alt="rooms of the house." />
<map id="map1" name="map1">
  <area shape="rect" coords="0,0,42,42"
    href="bedroom.html"/> <!-- missing alternative text attribute on <area> -->
  <area shape="rect" coords="0,0,21,21"
    href="lounge.html" alt=""/> <!-- empty `alt` attribute on <area> -->
</map>

<object {...props} />  <!-- missing alternative text attribute on <area> -->
```
<h4>Compliant solution</h4>

```js
<img src="foo.png" alt="Some textual description of foo.png" />
<input type="image" src="bar.png" aria-labelledby="Textual description of bar.png" />

<img src="house.gif" usemap="#map1"
    alt="rooms of the house." />
<map id="map1" name="map1">
  <area shape="rect" coords="0,0,42,42"
    href="bedroom.html" alt="Bedroom" />
  <area shape="rect" coords="0,0,21,21"
    href="lounge.html" aria-label="Lounge"/>
</map>

<object>My welcoming Bar</object>
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> W3C - <a href="https://www.w3.org/WAI/tutorials/images/decision-tree/">W3C WAI&nbsp;Web Accessibility Tutorials</a> </li>
  <li> W3C - <a href="https://www.w3.org/TR/WCAG20-TECHS/H24.html">Providing text alternatives for the area elements of image maps</a> </li>
  <li> W3C - <a href="https://www.w3.org/TR/WCAG20-TECHS/H36.html">Using alt attributes on images used as submit buttons</a> </li>
  <li> W3C - <a href="https://www.w3.org/TR/WCAG20-TECHS/H37.html">Using alt attributes on img elements</a> </li>
  <li> W3C - <a href="https://www.w3.org/TR/WCAG20-TECHS/H67.html">Using null alt text and no title attribute on img elements for images that AT
  should ignore</a> </li>
  <li> W3C - <a href="https://www.w3.org/TR/WCAG20-TECHS/H2.html">Combining adjacent image and text links for the same resource</a> </li>
  <li> W3C - <a href="https://www.w3.org/WAI/WCAG21/quickref/?versions=2.0#qr-text-equiv-all">Non-text Content</a> </li>
  <li> W3C - <a href="https://www.w3.org/WAI/WCAG21/quickref/?versions=2.0#qr-navigation-mechanisms-refs">Link Purpose (In Context)</a> </li>
  <li> W3C - <a href="https://www.w3.org/WAI/WCAG21/quickref/?versions=2.0#qr-navigation-mechanisms-link">Link Purpose (Link Only)</a> </li>
</ul>