<h2>Why is this an issue?</h2>
<p>Kotlin features language support for the delegator pattern using <code>by</code> clauses. Because this is a built-in language feature, it should be
used as an idiom instead of resorting to custom idioms.</p>
<h3>What is the potential impact?</h3>
<h4>Readability and Understanding</h4>
<p>This change makes it easier to understand the code because this is how delegation is intended to be used in Kotlin. When developers share common
standards and idioms, they need to spend less effort understanding each other’s code.</p>
<h4>Code Redundancy</h4>
<p>Using a built-in language feature or a standard API is always better than a custom implementation, because the reimplementation of something that
already exists is unnecessary.</p>
<h2>How to fix it</h2>
<p>Remove all <code>override</code> functions from your class that delegate to the function with the same signature in the delegee. In the class
header, add a <code>by</code> clause to delegate the entire interface to the delegee.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```kotlin
interface Network { fun connect() }

class PoorNetwork : Network {
    override fun connect() { println("cannot connect") }
}

class GoodNetwork : Network {
    override fun connect() { println("connected") }
}

interface Graphics { fun render() }

class Nvidia : Graphics {
    override fun render() { println("Neat 3D world") }
}

class AsusCardFrom2010 : Graphics {
    override fun render() { println("~8 fps") }
}

abstract class OS : Network, Graphics
```

```kotlin
class Linux : OS() {
    private val network = GoodNetwork()
    override fun connect() = network.connect() // Noncompliant, explicit function delegation
    private val graphics= Nvidia()
    override fun render() = graphics.render() // Noncompliant, explicit function delegation
    // ...
}

class Windows : OS() {
    private val network = PoorNetwork()
    override fun connect() = network.connect() // Noncompliant, explicit function delegation
    private val graphics = AsusCardFrom2010()
    override fun render() = graphics.render() // Noncompliant, explicit function delegation
    // ...
}
```
<h4>Compliant solution</h4>

```kotlin
class Linux:
    OS(),
    Network by GoodNetwork(), // Compliant, interface delegation using `by`
    Graphics by Nvidia() // Compliant, interface delegation using `by`
{
    // ...
}
class Windows:
    OS(),
    Network by PoorNetwork(), // Compliant, interface delegation using `by`
    Graphics by AsusCardFrom2010() // Compliant, interface delegation using `by`
{
    // ...
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://kotlinlang.org/docs/delegation.html">Kotlin Docs, Delegation</a> </li>
</ul>