<p>This vulnerability makes it possible that an encrypted communication is intercepted.</p>
<h2>Why is this an issue?</h2>
<p>Transport Layer Security (TLS) provides secure communication between systems over the internet by encrypting the data sent between them.
Certificate validation adds an extra layer of trust and security to this process to ensure that a system is indeed the one it claims to be.</p>
<p>When certificate validation is disabled, the client skips a critical security check. This creates an opportunity for attackers to pose as a trusted
entity and intercept, manipulate, or steal the data being transmitted.</p>
<h3>What is the potential impact?</h3>
<p>Establishing trust in a secure way is a non-trivial task. When you disable certificate validation, you are removing a key mechanism designed to
build this trust in internet communication, opening your system up to a number of potential threats.</p>
<h4>Identity spoofing</h4>
<p>If a system does not validate certificates, it cannot confirm the identity of the other party involved in the communication. An attacker can
exploit this by creating a fake server and masquerading as a legitimate one. For example, they might set up a server that looks like your bank’s
server, tricking your system into thinking it is communicating with the bank. This scenario, called identity spoofing, allows the attacker to collect
any data your system sends to them, potentially leading to significant data breaches.</p>
<h4>Loss of data integrity</h4>
<p>When TLS certificate validation is disabled, the integrity of the data you send and receive cannot be guaranteed. An attacker could modify the data
in transit, and you would have no way of knowing. This could range from subtle manipulations of the data you receive to the injection of malicious
code or malware into your system. The consequences of such breaches of data integrity can be severe, depending on the nature of the data and the
system.</p>
<h2>How to fix it in Node.js</h2>
<h3>Code examples</h3>
<p>The following code contains examples of disabled certificate validation.</p>
<p>The certificate validation gets disabled by setting <code>rejectUnauthorized</code> to <code>false</code>. To enable validation set the value to
<code>true</code> or do not set <code>rejectUnauthorized</code> at all to use the secure default value.</p>
<h4>Noncompliant code example</h4>

```ts
const https = require('node:https');

let options = {
  hostname: 'www.example.com',
  port: 443,
  path: '/',
  method: 'GET',
  rejectUnauthorized: false,
  secureProtocol: 'TLSv1_2_method'
};

let req = https.request(options, (res) => {
  res.on('data', (d) => {
    process.stdout.write(d);
  });
}); // Noncompliant
```

```ts
const tls = require('node:tls');

let options = {
    rejectUnauthorized: false,
    secureProtocol: 'TLSv1_2_method'
};

let socket = tls.connect(443, "www.example.com", options, () => {
  process.stdin.pipe(socket);
  process.stdin.resume();
}); // Noncompliant
```
<h4>Compliant solution</h4>

```ts
const https = require('node:https');

let options = {
  hostname: 'www.example.com',
  port: 443,
  path: '/',
  method: 'GET',
  secureProtocol: 'TLSv1_2_method'
};

let req = https.request(options, (res) => {
  res.on('data', (d) => {
    process.stdout.write(d);
  });
});
```

```ts
const tls = require('node:tls');

let options = {
    secureProtocol: 'TLSv1_2_method'
};

let socket = tls.connect(443, "www.example.com", options, () => {
  process.stdin.pipe(socket);
  process.stdin.resume();
});
```
<h3>How does this work?</h3>
<p>Addressing the vulnerability of disabled TLS certificate validation primarily involves re-enabling the default validation.</p>
<p>To avoid running into problems with invalid certificates, consider the following sections.</p>
<h4>Using trusted certificates</h4>
<p>If possible, always use a certificate issued by a well-known, trusted CA for your server. Most programming environments come with a predefined list
of trusted root CAs, and certificates issued by these authorities are validated automatically. This is the best practice, and it requires no
additional code or configuration.</p>
<h4>Working with self-signed certificates or non-standard CAs</h4>
<p>In some cases, you might need to work with a server using a self-signed certificate, or a certificate issued by a CA not included in your trusted
roots. Rather than disabling certificate validation in your code, you can add the necessary certificates to your trust store.</p>
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/Top10/A05_2021-Security_Misconfiguration/">Top 10 2021 Category A5 - Security Misconfiguration</a> </li>
  <li> OWASP - <a href="https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/">Top 10 2021 Category A7 - Identification and
  Authentication Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">Top 10 2017 Category A6 - Security
  Misconfiguration</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m3-insecure-communication">Mobile Top 10 2016 Category M3 - Insecure
  Communication</a> </li>
  <li> OWASP - <a href="https://mas.owasp.org/checklists/MASVS-NETWORK/">Mobile AppSec Verification Standard - Network Communication Requirements</a>
  </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/295">CWE-295 - Improper Certificate Validation</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222550">Application Security and
  Development: V-222550</a> - The application must validate certificates by constructing a certification path to an accepted trust anchor. </li>
</ul>