<h2>Why is this an issue?</h2>
<p>In React, <code>useState</code> is a hook that allows functional components to manage and update state in a manner similar to class components.
When you use the <code>useState</code> hook, it returns an array with two values: the current state value and a function to update that state
value.</p>
<p>Destructuring these values and naming them symmetrically (i.e., using consistent variable names for both the current state and the update function)
is a recommended best practice:</p>
<ul>
  <li> When you destructure and name the values symmetrically, it makes your code more readable and self-explanatory. Other developers can quickly
  understand the purpose of each variable without needing to refer back to the <code>useState</code> function call. </li>
  <li> Following a naming convention where the state variable and its corresponding update function have similar names is a common practice in the
  React community. It helps maintain consistency and makes it easier for others to understand your code. </li>
  <li> If you don’t name the variables symmetrically, it can lead to confusion, especially in larger components or when multiple state variables are
  involved. You might accidentally use the wrong variable when updating the state, which can result in bugs that are hard to track down. </li>
</ul>

```ts
import { useState } from 'react';
function MyComponent() {
  const [count, update] = useState(0); // Noncompliant
  return <div onClick={() => update(count + 1)}>{count}</div>
}
```
<p>You should destructure the return value of <code>useState</code> calls in terms of the current state and a function to update that state and name
them symmetrically.</p>

```ts
import { useState } from 'react';
function MyComponent() {
  const [count, setCount] = useState(0);
  return <div onClick={() => setCount(count + 1)}>{count}</div>
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/reference/react/useState">useState</a> </li>
</ul>