"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_NAME = void 0;
const utils_1 = require("@angular-eslint/utils");
const utils_2 = require("@typescript-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
exports.RULE_NAME = 'sort-ngmodule-metadata-arrays';
const DEFAULT_LOCALE = 'en-US';
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'suggestion',
        docs: {
            description: 'Ensures ASC alphabetical order for `NgModule` metadata arrays for easy visual scanning',
        },
        fixable: 'code',
        schema: [
            {
                type: 'object',
                properties: {
                    locale: {
                        type: 'string',
                        description: 'A string with a BCP 47 language tag.',
                        default: DEFAULT_LOCALE,
                    },
                },
                additionalProperties: false,
            },
        ],
        messages: {
            sortNgmoduleMetadataArrays: '`NgModule` metadata arrays should be sorted in ASC alphabetical order',
        },
    },
    defaultOptions: [
        {
            locale: DEFAULT_LOCALE,
        },
    ],
    create(context, [{ locale }]) {
        return {
            [`${utils_1.Selectors.MODULE_CLASS_DECORATOR} Property[key.name!="deps"] > ArrayExpression`]({ elements, }) {
                const unorderedNodes = elements
                    .filter(utils_2.ASTUtils.isIdentifier)
                    .map((current, index, list) => [current, list[index + 1]])
                    .find(([current, next]) => {
                    return next && current.name.localeCompare(next.name, locale) === 1;
                });
                if (!unorderedNodes)
                    return;
                const [unorderedNode, nextNode] = unorderedNodes;
                context.report({
                    node: nextNode,
                    messageId: 'sortNgmoduleMetadataArrays',
                    fix: (fixer) => [
                        fixer.replaceText(unorderedNode, nextNode.name),
                        fixer.replaceText(nextNode, unorderedNode.name),
                    ],
                });
            },
        };
    },
});
