External `FORM` calls introduce dependencies that are not validated at compile time, potentially leading to more fragile and complex code. 
Consider replacing `FORM` calls with proper modularization units like classes and methods.

# What Does This Check Look For?

This check reports a finding when subroutines (`FORM`) are called from external programs using the construct `PERFORM ... IN PROGRAM ...`.

# Why is This a Problem?

Calling external `FORM` routines can introduce several issues:

- **No Compile-Time Validation**:
  The compiler cannot verify whether the external `FORM` exists, has the correct signature, or whether the parameters match. 
  Errors only surface at runtime.
- **Hidden Dependencies**:
  The Where-Used List does not reliably detect these calls.
  When the external program is modified or the `FORM` is deleted, the code can break without a warning.
- **Violation of Encapsulation**:
  `FORM` routines are internal building blocks of a program and are not intended for external use.
  Calling them from outside violates the principle of information hiding.
- **Maintenance Nightmare**:
  The dependencies are not documented and, hence, difficult to trace.
  This can make the code harder to understand and maintain.
- **No Interface Contract**: Unlike Function Modules or methods, `FORM` routines have no formal interface definition.
  Changes to parameters in the external program can silently break the code.

By avoiding external `FORM` calls, the code becomes more robust, maintainable, and easier to understand.

# How Can I Resolve This?

Replace external `FORM` calls with proper modularization units that provide a stable and documented interface.

Use **classes and methods** for reusable logic that needs to be called from multiple programs.
They offer compile-time parameter checking and appear in the Where-Used List.
With their clean, object-oriented design, they also provide strong typing, better encapsulation, and improved testability, even compared to function modules.

If you need to extend SAP standard functionality, use **Business Add-Ins (BAdIs)** , **Enhancement Points** , or **User Exits** instead of calling a `FORM` defined in a customer program.

# Examples
## Non-Compliant

```ABAP
" non-compliant: external FORM call
PERFORM get_customer_data IN PROGRAM my_report
    USING customer_id
    CHANGING customer.

" non-compliant: dynamic external FORM call
PERFORM (form_name) IN PROGRAM (program_name).
```

## Compliant

```ABAP
" compliant: function module call
CALL FUNCTION 'GET_CUSTOMER'
    EXPORTING customer_id = customer_id
    IMPORTING customer = customer.

" compliant: method call
customer = customer_service->get_customer( customer_id ).

" compliant: local FORM within the same program
PERFORM get_customer
    USING customer_id
    CHANGING customer.
```

# Where Can I Learn More?
- [ABAP Keyword Documentation | PERFORM](https://help.sap.com/doc/abapdocu_latest_index_htm/latest/en-US/ABAPPERFORM_FORM.html)


