# Dir 4.6 typedefs that indicate size and signedness should be used in place of the basic numerical types 

## Category

Advisory 

## Applies to

C90, C99, C11 

## Amplification

The basic numerical types of `char`, `short`, `int`, `long`, `long long`, `float`, `double` and `long double` should not be used, but specific-length typedefs should be used. The numerical types of `char` are `signed char` and `unsigned char`. These Guidelines do not treat "plain" `char` as a numerical type (see Section 8.10.2 on essentially character types). 

For C99 and later, the types provided by `<stdint.h>` should be used. For C90, equivalent types should be defined and used. 

A type must not be defined with a specific length unless the implemented type is actually of that length. It is not necessary to use typedefs in the declaration of bit-fields. 

For example, on a C90 implementation (where the standard header `<stdint.h>` is not available) the following definitions may be suitable, but need to be adjusted to suit the project's implementation: 

```c
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed int int32_t;
typedef signed long int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned int uint32_t;
typedef unsigned long uint64_t;
typedef float float32_t;
typedef double float64_t;
typedef long double float128_t;

```

## Rationale

In situations where the amount of memory being allocated is important, using specific-length types makes it clear how much storage is being reserved for each object. 

Adherence to this guideline does not guarantee portability because the size of the `int` type may determine whether or not an expression is subject to integer promotion. For example, an expression with type `int16_t` will not be promoted if `int` is implemented using 16 bits but will be promoted if `int` is implemented using 32 bits. This is discussed in more detail in the section on integer promotion in Appendix C. 

Note: defining a specific-length type whose size is not the same as the implemented type is counter-productive both in terms of storage requirements and in terms of portability. Care should be taken to avoid defining types with the wrong size. 

If abstract types are defined in terms of a specific-length type then it is not necessary, and may even be undesirable, for those abstract types to specify the size or sign. For example, the following code defines an abstract type representing mass in kilograms but does not indicate its size or sign: `typedef uint16_t mass_kg_t;` 

It might be desirable not to apply this guideline when interfacing with the Standard Library or code outside the project's control. 

## Exception
1. The basic numerical types may be used in a typedef to define a specific-length type. 
2. For function `main` an `int` may be used rather than the typedefs as a return type. Therefore `int main (void)` is permitted. 
3. For function `main` an `int` may be used rather than the typedefs for the input parameter `argc`. Therefore `int main(int argc, char *argv[])` is permitted. 

## Example

```c
/* Non-compliant int used to define an object */
int x = 0;

/* Compliant int used to define specific-length type */
typedef int SINT_16;

/* Non-compliant no sign or size specified */
typedef int speed_t;

/* Compliant further abstraction does not need specific length */
typedef int16_t torque_t;

```

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
