# Dir 5.1 There shall be no data races between threads

## Category
Required

## Applies to
C11

## Amplification
Two expression evaluations conflict if one of them modifies a memory location and the other one reads or modifies the same memory location. The execution of a program contains a data race if it contains two conflicting actions in different threads, at least one of which is not atomic, and neither happens before the other, i.e. there is no fixed ordering between the two actions. To prevent data races, objects shared between different threads shall be protected by an appropriate synchronization mechanism.

## Rationale
Data races are caused by simultaneous accesses to the same non-atomic object from two different threads T1 and T2 where at least one of them is a write access and where the program semantics does not impose a fixed ordering between T1 and T2. There may be legitimate program executions where the access from T1 is executed before the access from thread T2, and vice versa, or where a given access itself is interrupted. Any such data race results in undefined behaviour.

There are several critical scenarios:
*   Depending on the timing of the threads, sometimes in a given context the wrong value might be used, leading to unexpected results.
*   If a read or write access is implemented by several machine instructions, a pre-emption might occur between these instructions such that inconsistent values might be read or written. As an example, a 64-bit variable read implemented as two 32-bit load instructions might be interrupted after reading the first 32 bits. Then another thread might change the variable value. When the first thread resumes, it reads the second 32-bit half, which now contains a different value than when the first 32 bits of the variable were read.

In general, a data race can cause memory corruption and lead to unexpected, erroneous or erratic behaviour. Data races typically manifest sporadically and are very hard to reproduce.

To prevent such situations, when an object is shared between different threads, it shall be protected by an appropriate synchronization mechanism. To ensure consistent access within a single shared object it can be declared as atomic. A more general solution to ensure consistency of accesses is to introduce critical sections with mutex locks or condition variables.

Note: C Standard Library functions may access objects with static or thread storage duration directly or indirectly via the function’s arguments. The C Standard Library functions *setlocale*, *tmpnam*, *rand*, *srand*, *getenv*, *getenv_s*, *strtok*, *strerror*, *asctime*, *ctime*, *gmtime*, *localtime*, *mbrtoc16*, *c16rtomb*, *mbrtoc32*, *c32rtomb*, *mbrlen*, *mbrtowc*, *wcrtomb*, *mbsrtowcs*, *wcsrtombs* are not guaranteed to be reentrant and may modify objects with static or thread storage duration. To prevent data races explicit synchronization may be required.

## Example
The following example exhibits data races on the global variables `x` and `a`. Functions `t1`, `t2`, `t3` and `t4` are executed as concurrent threads T1, T2, T3 and T4 respectively.

Variable `x` is accessed without synchronization, by function `t1` in thread T1 and by function `t2` in thread T2. If executed on a 16-bit machine writing 32-bit values in two chunks of 16 bits, threads T1 and T2 might interrupt one another after the first 16 bits of the variable have been written. As a consequence, the two 16-bit halves of variable `x` might be written by different threads, causing unexpected values.

```c
int32_t x;
int32_t a = 1;
int32_t b;

int32_t t1( void *ignore ) /* Thread T1 entry */
{
  while ( 1 ) 
  {
    x = -1; /* Write-write data race with t2. Possible values of x: 0xFFFF0000, 
               0x0000FFFF, 0x00000000, 0xFFFFFFFF */
  }
  return 0;
}

int32_t t2( void *ignore ) /* Thread T2 entry */
{
  while ( 1 )
  {
    x = 0;  /* Write-write data race with t1. Possible values of x: 0xFFFF0000, 
               0x0000FFFF, 0x00000000, 0xFFFFFFFF */
  }
  return 0;
}
```

A data race on `a` is caused by unprotected accesses by function `t3` in thread T3 and by function `t4` in thread T4. If thread T3 sees the value of 1 in variable `a`, it will enter the then-part of the conditional statement. At that point, it might be interrupted by thread T4, which sets `a` to 0. After resuming, thread T3 will run into a division by zero.

```c
int32_t t3( void *ignore ) /* Thread T3 entry */
{
  while ( 1 ) 
  {
    if ( a != 0 ) /* Read-write data race with T4 */
    { 
      b += 1/a;   /* Read-write data race with T4 */
      a = 1;      /* Write-write data race with T4 */
    }
  }
  return 0;
}

int32 t4( void *ignore ) /* Thread T4 entry */
{
  while ( 1 ) 
  {
    a = 0;        /* Read-write data race with T3 */
  }
  return 0;
}
```

## See also
Rule 9.7, Rule 12.6

---

Copyright The MISRA Consortium Limited © [Date - March 2025].