# Rule 18.1 A pointer resulting from arithmetic on a pointer operand shall address an element of the same array as that pointer operand

## Category
Required

## Analysis
Undecidable, System

## Applies to
C90, C99

## Amplification
Creation of a pointer to one beyond the end of the array is well-defined by The Standard and is
permitted by this rule. Dereferencing a pointer to one beyond the end of an array results in undefined
behaviour and is forbidden by this rule.

This rule applies to all forms of array indexing:
```c
integer_expression + pointer_expression
pointer_expression + integer_expression
pointer_expression - integer_expression
pointer_expression += integer_expression
pointer_expression -= integer_expression
++pointer_expression
pointer_expression++
--pointer_expression
pointer_expression--
pointer_expression [ integer_expression ]
integer_expression [ pointer_expression ]
```
*Note*: a subarray is also an array.

*Note*: for purposes of pointer arithmetic, The Standard treats an object that is not a member of an
array as if it were an array with a single element (C90 Section 6.3.6, C99 Section 6.5.6).

## Rationale
Although some compilers may be able to determine at compile time that an array boundary has been
exceeded, no checks are generally made at run-time for invalid array subscripts. Using an invalid array
subscript can lead to erroneous behaviour of the program.

Run-time derived array subscript values are of the most concern since they cannot easily be checked
by static analysis or manual review. Code of a defensive programming nature should, where possible
and practicable, be provided to check such subscript values against valid ones and, if required,
appropriate action be taken.

It is undefined behaviour if the result obtained from one of the above expressions is not a pointer to
an element of the array pointed to by `pointer_expression` or an element one beyond the end of
that array. See C90 Section 6.3.6 and C99 Section 6.5.6 for further information.

Multi-dimensional arrays are "arrays of arrays". This rule does not allow pointer arithmetic that results
in the pointer addressing a different subarray. Array subscripting over "internal" boundaries shall not
be used, as such behaviour is undefined.

## Example
The use of the `+` operator will also violate Rule 18.4.
```c
int32_t f1 ( int32_t * const a1, int32_t a2[ 10 ] )
{
 int32_t *p = &a1[ 3 ]; /* Compliant/non-compliant depending on
 * the value of a1 */
 return *( a2 + 9 ); /* Compliant */
}
void f2 ( void )
{
 int32_t data = 0;
 int32_t b = 0;
 int32_t c[ 10 ] = { 0 };
 int32_t d[ 5 ][ 2 ] = { 0 }; /* 5-element array of 2-element arrays
 * of int32_t */
int32_t *p1 = &c[ 0 ]; /* Compliant */
int32_t *p2 = &c[ 10 ]; /* Compliant - points to one beyond */
int32_t *p3 = &c[ 11 ]; /* Non-compliant - undefined, points to
 * two beyond */
data = *p2; /* Non-compliant - undefined, dereference
 * one beyond */
data = f1 ( &b, c );
data = f1 ( c, c );
p1++; /* Compliant */
c[ -1 ] = 0; /* Non-compliant - undefined, array
 * bounds exceeded */
data = c[ 10 ]; /* Non-compliant - undefined, dereference
 * of address one beyond */
data = *( &data + 0 ); /* Compliant - C treats data as an
 * array of size 1 */
d[ 3 ][ 1 ] = 0; /* Compliant */
data = *( *( d + 3 ) + 1 ); /* Compliant */
data = d[ 2 ][ 3 ]; /* Non-compliant - undefined, internal
 * boundary exceeded */
p1 = d[ 1 ]; /* Compliant */
data = p1[ 1 ]; /* Compliant - p1 addresses an array
 * of size 2 */
}
```
The following example illustrates pointer arithmetic applied to members of a structure. Because each
member is an object in its own right, this rule prevents the use of pointer arithmetic to move from one
member to the next. However, it does not prevent arithmetic on a pointer to a member provided that
the resulting pointer remains within the bounds of the member object.
```c
struct
{
 uint16_t x;
 uint16_t y;
 uint16_t z;
 uint16_t a[ 10 ];
} s;
uint16_t *p;
void f3 ( void )
{
 p = &s.x;
 ++p; /* Compliant - p points one beyond s.x */
 p[ 0 ] = 1; /* Non-compliant - undefined, dereference of address one
 * beyond s.x which is not necessarily
 * the same as s.y */
 p[ 1 ] = 2; /* Non-compliant - undefined */
p = &s.a[ 0 ]; /* Compliant - p points into s.a */
p = p + 8; /* Compliant - p still points into s.a */
p = p + 3; /* Non-compliant - undefined, p points more than one
 * beyond s.a */
}
```

## See also
Dir 4.1, Rule 18.4

---

Copyright The MISRA Consortium Limited © [Date - March 2013].
