# Rule 22.12 Thread objects, thread synchronization objects, and thread-specific storage pointers shall only be accessed by the appropriate Standard Library functions

## Category
Mandatory

## Analysis
Undecidable, System

## Applies to
C11

## Amplification
Thread objects shall exclusively be accessed via the Standard Library functions *thrd_create()*,
*thrd_detach()*, *thrd_join()*, and *thrd_equal()*.

Mutex objects shall exclusively be accessed via the Standard Library functions *mtx_destroy()*, *mtx_init()*,
*mtx_lock()*, *mtx_trylock()*, *mtx_timedlock()*, *mtx_unlock()*, *cnd_wait()*, and *cnd_timedwait()*.

Condition variables shall exclusively be accessed via the Standard Library functions *cnd_broadcast()*,
*cnd_destroy()*, *cnd_init()*, *cnd_signal()*, *cnd_wait()*, and *cnd_timedwait()*.

Thread-specific storage pointers shall exclusively be accessed by the Standard Library functions
*tss_create()*, *tss_delete()*, *tss_get()*, and *tss_set()*.

## Rationale
Thread objects and thread synchronization objects are expected to be unique for the corresponding
thread and synchronization resources.

Thread-specific storage pointers are identified by unique keys. Any direct manipulation (copy,
assignment, etc.) may result in undefined behaviour. The *tss_delete()*, *tss_get()* and *tss_set()* functions
shall only be called with a value for key that was returned by a call to *tss_create()*, otherwise the
behaviour is undefined.

## Example
```c
mtx_t Ra;
mtx_t Rb;
thrd_t id1;
thrd_t id2;
tss_t key;

int32_t t1( void *ignore )
{
 mtx_lock( &Ra );
 int32_t val;
 if ( id1 == id2 ) /* Non-compliant - use thrd_equal() */
 {
 Rb = Ra; /* Non-compliant */
 memcpy(&Rb, &Ra, sizeof(mtx_t)); /* Non-compliant */
 }
 if ( thrd_equal( id1, id2 ) ) /* Compliant */
 {
 ...
 }
 key++; /* Non-compliant - explicit manipulation of
 TSS pointer */
 tss_set( key, &val ); /* Undefined, value of key not returned by
 tss_create() */
}

void main( void )
{
 mtx_init ( &Ra, mtx_plain );
 mtx_init ( &Rb, mtx_plain );
 tss_create ( &key, NULL );
 thrd_create( &id1, t1, NULL );
 thrd_create( &id2, t1, NULL );
 ...
}
```

## See also
Rule 11.5, Rule 22.20

---

Copyright The MISRA Consortium Limited © [Date - April 2023].
