# Rule 8.2.2 C-style casts and *functional notation* casts shall not be used

## Category
Required

## Analysis
Decidable, Single Translation Unit

## Amplification
This rule does not apply to *functional notation* casts that use curly braces or that result in a constructor
call.

## Rationale
C-style casts and *functional notation* casts raise several concerns:
1.  They permit almost any type to be converted to almost any other type without checks;
2.  They give no indication why the conversion is taking place;
3.  Their syntax is more difficult to recognize.

These concerns can be addressed with the use of `const_cast`, `dynamic_cast`, `static_cast` and
`reinterpret_cast`, which:
1.  Enforce constraints on the types involved;
2.  Give a better indication of the cast’s intent;
3.  Are easy to identify.

## Exception
A C-style cast to `void` is permitted, as this allows the intentional discarding of a value to be made
explicit — for instance, the return value of a non-void function call (see Rule 0.1.2).

## Example
```cpp
int32_t g();

void f1()
{
  ( void ) g(); // Compliant by exception
}
```
In the following example (which violates Rule 8.2.3), the C-style casts from `a1` to the non-const pointer
`a2` is more permissive than necessary. If the type of `a1` is not `A`, then the C-style cast to `a2` will compile,
resulting in *undefined behaviour*. The equivalent `const_cast` to `a3` will not compile if the type of `a1`
is changed.
```cpp
struct A
{
  A( char c);
};

struct B {};

void f1a( A x )
{
  auto const & a1 = x;
  A * a2 = ( A * )&a1; // Non-compliant
  A * a3 = const_cast< A * >( &a1 );
}

void f1b( B x )
{
  auto const & a1 = x;
  A * a2 = ( A * )&a1; // Non-compliant
  A * a3 = const_cast< A * >( &a1 ); // Ill-formed
}

void f2( int32_t x )
{
  auto i = A( 'c' ); // Rule does not apply - constructor is called
  auto j = int8_t { 42 }; // Rule does not apply - use of curly braces
  auto k = int8_t ( x ); // Non-compliant - does not construct an object
} // of class type
```

## See also
Rule 0.1.2, Rule 8.2.3

---

Copyright The MISRA Consortium Limited © [Date - October 2023].
