<h2>Why is this an issue?</h2>
<p>Both the <code>List.Find</code> method and <code>IEnumerable.FirstOrDefault</code> method can be used to find the first element that satisfies a
given condition in a collection. However, <code>List.Find</code> can be faster than <code>IEnumerable.FirstOrDefault</code> for <code>List</code>
objects. For small collections, the performance difference may be minor, but for large collections, it can make a noticeable difference. The same
applies for <code>ImmutableList</code> and arrays too.</p>
<p><strong>Applies to</strong></p>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.find">List</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.find">Array</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.find">ImmutableList</a> </li>
</ul>
<h3>What is the potential impact?</h3>
<p>We measured at least 2x improvement in the execution time. For more details see the <code>Benchmarks</code> section from the <code>More info</code>
tab.</p>
<h2>How to fix it</h2>
<p>The <code>Find</code> method is defined on the collection class, and it has the same signature as <code>FirstOrDefault</code> extension method. The
function can be replaced in place.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
int GetValue(List&lt;int&gt; data) =&gt;
    data.FirstOrDefault(x =&gt; x % 2 == 0);
</pre>
<pre data-diff-id="2" data-diff-type="noncompliant">
int GetValue(int[] data) =&gt;
    data.FirstOrDefault(x =&gt; x % 2 == 0);
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
int GetValue(List&lt;int&gt; data) =&gt;
    data.Find(x =&gt; x % 2 == 0);
</pre>
<pre data-diff-id="2" data-diff-type="compliant">
int GetValue(int[] data) =&gt;
    Array.Find(data, x =&gt; x % 2 == 0);
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.find">List&lt;T&gt;.Find(Predicate&lt;T&gt;)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.find">Array.Find&lt;T&gt;(T[], Predicate&lt;T&gt;)</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.find">ImmutableList&lt;T&gt;.Find(Predicate&lt;T&gt;)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.firstordefault">Enumerable.FirstOrDefault(Predicate&lt;T&gt;)</a>
  </li>
</ul>
<h3>Benchmarks</h3>
<table>
  <colgroup>
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
  </colgroup>
  <thead>
    <tr>
      <th>Method</th>
      <th>Runtime</th>
      <th>Mean</th>
      <th>Standard Deviation</th>
      <th>Allocated</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td><p>FirstOrDefault</p></td>
      <td><p>.NET 7.0</p></td>
      <td><p>5.373 ms</p></td>
      <td><p>0.1049 ms</p></td>
      <td><p>125 KB</p></td>
    </tr>
    <tr>
      <td><p>Find</p></td>
      <td><p>.NET 7.0</p></td>
      <td><p>1.691 ms</p></td>
      <td><p>0.0334 ms</p></td>
      <td><p>85.94 KB</p></td>
    </tr>
    <tr>
      <td><p>FirstOrDefault</p></td>
      <td><p>.NET Framework 4.6.2</p></td>
      <td><p>5.035 ms</p></td>
      <td><p>0.0421 ms</p></td>
      <td><p>125.38 KB</p></td>
    </tr>
    <tr>
      <td><p>Find</p></td>
      <td><p>.NET Framework 4.6.2</p></td>
      <td><p>1.779 ms</p></td>
      <td><p>0.0107 ms</p></td>
      <td><p>86.2 KB</p></td>
    </tr>
  </tbody>
</table>
<h4>Glossary</h4>
<ul>
  <li> <a href="https://en.wikipedia.org/wiki/Arithmetic_mean">Mean</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Standard_deviation">Standard Deviation</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Memory_management">Allocated</a> </li>
</ul>
<p>The results were generated by running the following snippet with <a href="https://github.com/dotnet/BenchmarkDotNet">BenchmarkDotNet</a>:</p>
<pre>
private List&lt;string&gt; data;
private Random random = new Random();

[Params(1_000)]
public int N { get; set; }

[GlobalSetup]
public void Setup() =&gt;
    data = Enumerable.Range(0, N).Select(x =&gt; Guid.NewGuid().ToString()).ToList();

[Benchmark(Baseline = true)]
public void FirstOrDefault()
{
    for (var i = 0; i &lt; N; i++)
    {
        var value = data[random.Next(N - 1)];
        _ = data.FirstOrDefault(x =&gt; x == value);   // Enumerable.FirstOrDefault()
    }
}

[Benchmark]
public void Find()
{
    for (var i = 0; i &lt; N; i++)
    {
        var value = data[random.Next(N - 1)];
        _ = data.Find(x =&gt; x == value);             // List&lt;T&gt;.Find()
    }
}
</pre>
<p>Hardware configuration:</p>
<pre>
BenchmarkDotNet=v0.13.5, OS=Windows 10 (10.0.19045.2846/22H2/2022Update)
11th Gen Intel Core i7-11850H 2.50GHz, 1 CPU, 16 logical and 8 physical cores
  [Host]               : .NET Framework 4.8 (4.8.4614.0), X64 RyuJIT VectorSize=256
  .NET 7.0             : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
  .NET Framework 4.6.2 : .NET Framework 4.8 (4.8.4614.0), X64 RyuJIT VectorSize=256
</pre>

