<h2>Why is this an issue?</h2>
<p>When the same code is duplicated in two or more separate branches of a conditional, it can make the code harder to understand, maintain, and can
potentially introduce bugs if one instance of the code is changed but others are not.</p>
<p>Having two clauses in a <code>when</code> statement or two branches in an <code>if</code> chain with the same implementation is at best duplicate
code, and at worst a coding error.</p>

```kotlin
if (a >= 0 && a < 10) {
  doFirstThing()
  doTheThing()
}
else if (a >= 10 && a < 20) {
  doTheOtherThing()
}
else if (a >= 20 && a < 50) { // Noncompliant; duplicates first condition
  doFirstThing()
  doTheThing()
}
else {
  doTheRest()
}
```

```kotlin
when (x) {
  1 -> {
    doFirstThing()
    doSomething()
  }
  2 -> doSomethingDifferent()
  3 -> { // Noncompliant; duplicates case 1's implementation
    doFirstThing()
    doSomething()
  }
  else -> doTheRest()
}
```
<p>If the same logic is needed for both instances, then:</p>
<ul>
  <li> in an <code>if</code> structure they should be combined </li>
</ul>

```kotlin
if ((a >= 0 && a < 10) || (a >= 20 && a < 50)) {
  doFirstThing()
  doTheThing()
}
else if (a >= 10 && a < 20) {
  doTheOtherThing()
}
else {
  doTheRest()
}
```
<ul>
  <li> for a <code>when</code>, the values should be put together in the branch expression list. </li>
</ul>

```kotlin
when (x) {
  1, 3 -> {
    doFirstThing()
    doSomething()
  }
  2 -> doSomethingDifferent()
  else -> doTheRest()
}
```
<h3>Exceptions</h3>
<p>Blocks in an <code>if</code> chain or in a <code>when</code> branch that contain a single line of code are ignored.</p>

```kotlin
if (a == 1) {
    doSomething()  //no issue, usually this is done on purpose to increase the readability
} else if (a == 2) {
    doSomethingElse()
} else {
    doSomething()
}
```
<p>But this exception does not apply to <code>if</code> chains without <code>else</code>-s, or to <code>when</code>-es without <code>else</code>
clauses when all branches have the same single line of code. In case of <code>if</code> chains with <code>else</code>-s, or of <code>when</code>-es
with default clauses, rule <a href='checks/checks/?languages=KOTLIN&tools=SONAR_LINT&rule=kotlin:S3923'>kotlin:S3923</a> raises a bug.</p>

```kotlin
if (a == 1) {
  doSomething()  // Noncompliant, this might have been done on purpose but probably not
} else if (a == 2) {
  doSomething()
}
```
<h2>Resources</h2>
<h3>Related rules</h3>
<ul>
  <li> <a href='checks/checks/?languages=KOTLIN&tools=SONAR_LINT&rule=kotlin:S3923'>kotlin:S3923</a> - All branches in a conditional structure should not have exactly the same implementation </li>
</ul>